#ifdef __APPLE__
#include <GLUT/glut.h>
#include <OpenGL/gl.h>
#include <OpenGL/OpenGL.h>
#else
#include <GL/glut.h>
#include <GL/gl.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <math.h>

#include "Button.h"
#include "DisplayObjects.h"
#include "FunctionControls.h"
#include "Quaternion.h"
#include "QuaternionDisplay.h"
#include "QuaternionControls.h"
#include "Utilities.h"
#include "Vector.h"

enum {
	AUTO_NORMALIZE_BUTTON_INDEX,
	RESET_TO_IDENTITY_BUTTON_INDEX,
	SHOW_FUNCTIONS_BUTTON_INDEX,
	ANIMATE_BUTTON_INDEX,
	APPLY_NORMALIZE_BUTTON_INDEX,
	APPLY_INVERT_BUTTON_INDEX,
	APPLY_MULTIPLY_BUTTON_INDEX,
	APPLY_SLERP_BUTTON_INDEX,
	APPLY_ROTATE_BUTTON_INDEX,
	CURRENT_MULTIPLY_BUTTON_INDEX,
	CURRENT_SLERP_BUTTON_INDEX,
	CURRENT_ROTATE_BUTTON_INDEX,
	NUMBER_OF_BUTTONS
};

static Quaternion quaternion;
static Quaternion slerpStartQuaternion, slerpEndQuaternion;
static int slerpStartTime = 0;
static Quaternion objectQuaternion;
static int animateStartTime = 0;

static bool mouseIsDown = false;
static int lastMouseX = 0;
static int lastMouseY = 0;
static bool draggingNumberField = false;
static bool draggingFunctionField = false;

static Button buttons[NUMBER_OF_BUTTONS];
static Button * selectedButton;

static bool autoNormalize = true;
static bool showingFunctions = false;
static bool animate = false;

static void initGL() {
	GLint viewport[4];
	float matSpecular[] = {1.0f, 1.0f, 1.0f, 1.0f};
	float ambient[] = {0.1f, 0.1f, 0.1f, 1.0f};
	
#ifdef __APPLE__
	GLint VBL = 1;
	CGLSetParameter(CGLGetCurrentContext(), kCGLCPSwapInterval, &VBL);
#endif
	
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glEnable(GL_DEPTH_TEST);
	
	glEnable(GL_LIGHTING);
	glEnable(GL_NORMALIZE);
	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);
	
	glMaterialf(GL_FRONT, GL_SHININESS, 32.0);
	glMaterialfv(GL_FRONT, GL_SPECULAR, matSpecular);
	
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glGetIntegerv(GL_VIEWPORT, viewport);
	gluPerspective(60, (float) viewport[2] / (float) viewport[3], 1.5, 400.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

static void quaternionChangedCallback(Quaternion value) {
	quaternion = value;
	if (autoNormalize) {
		Quaternion_normalize(&quaternion);
	}
	QuaternionControls_setQuaternion(quaternion);
	glutPostRedisplay();
}

static void axisAngleChangedCallback(Vector axis, float angle) {
	if (angle < 0.0001f) angle = 0.0001f;
	if (angle > (M_PI * 2) - 0.0001f) angle = (M_PI * 2) - 0.0001f;
	quaternion = Quaternion_fromAxisAngle(axis, angle);
	if (autoNormalize) {
		Quaternion_normalize(&quaternion);
	}
	QuaternionControls_setQuaternion(quaternion);
	glutPostRedisplay();
}

static void initPlayground() {
	quaternion = Quaternion_identity();
	QuaternionControls_init(quaternionChangedCallback, axisAngleChangedCallback);
	QuaternionControls_setQuaternion(quaternion);
	FunctionControls_init();
	buttons[RESET_TO_IDENTITY_BUTTON_INDEX] = Button_create("Reset to identity", 5, 5, 142);
	buttons[SHOW_FUNCTIONS_BUTTON_INDEX] = Button_create("Functions ->", 157, 5, 102);
	buttons[ANIMATE_BUTTON_INDEX] = Button_create("Animate OFF", 269, 5, 94);
	buttons[AUTO_NORMALIZE_BUTTON_INDEX] = Button_create("Auto-normalize ON", 373, 5, 150);
	buttons[APPLY_NORMALIZE_BUTTON_INDEX] = Button_create("Apply", 5, 114, 46);
	buttons[APPLY_INVERT_BUTTON_INDEX] = Button_create("Apply", 5, 93, 46);
	buttons[APPLY_MULTIPLY_BUTTON_INDEX] = Button_create("Apply", 5, 72, 46);
	buttons[APPLY_SLERP_BUTTON_INDEX] = Button_create("Apply", 5, 51, 46);
	buttons[APPLY_ROTATE_BUTTON_INDEX] = Button_create("Apply", 5, 30, 46);
	buttons[CURRENT_MULTIPLY_BUTTON_INDEX] = Button_create("<- Current", 807, 72, 86);
	buttons[CURRENT_SLERP_BUTTON_INDEX] = Button_create("<- Current", 807, 51, 86);
	buttons[CURRENT_ROTATE_BUTTON_INDEX] = Button_create("<- Current", 807, 30, 86);
}

static void displayFunc() {
	float lightPosition0[] = {-1.0f, 2.0f, 0.0f, 0.0f}, lightPosition1[] = {2.0f, -5.0f, 2.0f, 0.0f};
	float lightDiffuse[] = {0.85f, 0.85f, 0.85f, 1.0f};
	GLint viewport[4];
	int currentTime;
	
	currentTime = glutGet(GLUT_ELAPSED_TIME);
	glGetIntegerv(GL_VIEWPORT, viewport);
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	glLoadIdentity();
	glTranslatef(0.0f, 0.0f, -5.5f);
	
	glLightfv(GL_LIGHT0, GL_POSITION, lightPosition0);
	glLightfv(GL_LIGHT1, GL_POSITION, lightPosition1);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, lightDiffuse);
	
	if (animate) {
		Quaternion nextObjectQuaternion;
		float alpha;
		float magnitude;
		
		magnitude = sqrt((quaternion.x * quaternion.x) +
		                 (quaternion.y * quaternion.y) +
		                 (quaternion.z * quaternion.z) +
		                 (quaternion.w * quaternion.w));
		
		alpha = (currentTime - animateStartTime) / 500.0f;
		animateStartTime = currentTime;
		while (alpha > 1.0f) {
			alpha -= 1.0f;
			objectQuaternion = Quaternion_multiplied(objectQuaternion, quaternion);
		}
		nextObjectQuaternion = Quaternion_multiplied(objectQuaternion, quaternion);
		objectQuaternion = Quaternion_slerp(objectQuaternion, nextObjectQuaternion, alpha);
		
		if (!autoNormalize) {
			objectQuaternion.x *= magnitude;
			objectQuaternion.y *= magnitude;
			objectQuaternion.z *= magnitude;
			objectQuaternion.w *= magnitude;
		}
	} else {
		objectQuaternion = quaternion;
	}
	
	glViewport(0, 0, viewport[2] / 3, viewport[3]);
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glTranslatef(0.0f, 0.15f, 0.0f);
	gluPerspective(60, (float) viewport[2] / 3 / (float) viewport[3], 1.5, 400.0);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	drawWireMesh();
	applyQuaternion(objectQuaternion);
	drawAxisColoredSphere();
	glPopMatrix();
	
	glViewport(viewport[2] / 3, 0, viewport[2] / 3, viewport[3]);
	drawWireMesh();
	drawQuaternion(quaternion);
	
	glViewport(viewport[2] / 3 * 2, 0, viewport[2] / 3, viewport[3]);
	glPushMatrix();
	drawWireMesh();
	applyQuaternion(objectQuaternion);
	drawAxisCross();
	glPopMatrix();
	
	glViewport(0, 0, viewport[2], viewport[3]);
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, viewport[2], 0, viewport[3], -1, 1);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glDisable(GL_LIGHTING);
	glDisable(GL_DEPTH_TEST);
	
	if (showingFunctions) {
		FunctionControls_draw();
		Button_draw(&buttons[RESET_TO_IDENTITY_BUTTON_INDEX], selectedButton == &buttons[RESET_TO_IDENTITY_BUTTON_INDEX]);
		Button_draw(&buttons[SHOW_FUNCTIONS_BUTTON_INDEX], selectedButton == &buttons[SHOW_FUNCTIONS_BUTTON_INDEX]);
		Button_draw(&buttons[ANIMATE_BUTTON_INDEX], selectedButton == &buttons[ANIMATE_BUTTON_INDEX]);
		Button_draw(&buttons[APPLY_NORMALIZE_BUTTON_INDEX], selectedButton == &buttons[APPLY_NORMALIZE_BUTTON_INDEX]);
		Button_draw(&buttons[APPLY_INVERT_BUTTON_INDEX], selectedButton == &buttons[APPLY_INVERT_BUTTON_INDEX]);
		Button_draw(&buttons[APPLY_MULTIPLY_BUTTON_INDEX], selectedButton == &buttons[APPLY_MULTIPLY_BUTTON_INDEX]);
		Button_draw(&buttons[APPLY_SLERP_BUTTON_INDEX], selectedButton == &buttons[APPLY_SLERP_BUTTON_INDEX]);
		Button_draw(&buttons[APPLY_ROTATE_BUTTON_INDEX], selectedButton == &buttons[APPLY_ROTATE_BUTTON_INDEX]);
		Button_draw(&buttons[CURRENT_MULTIPLY_BUTTON_INDEX], selectedButton == &buttons[CURRENT_MULTIPLY_BUTTON_INDEX]);
		Button_draw(&buttons[CURRENT_SLERP_BUTTON_INDEX], selectedButton == &buttons[CURRENT_SLERP_BUTTON_INDEX]);
		Button_draw(&buttons[CURRENT_ROTATE_BUTTON_INDEX], selectedButton == &buttons[CURRENT_ROTATE_BUTTON_INDEX]);
	} else {
		QuaternionControls_draw();
		Button_draw(&buttons[AUTO_NORMALIZE_BUTTON_INDEX], selectedButton == &buttons[AUTO_NORMALIZE_BUTTON_INDEX]);
		Button_draw(&buttons[RESET_TO_IDENTITY_BUTTON_INDEX], selectedButton == &buttons[RESET_TO_IDENTITY_BUTTON_INDEX]);
		Button_draw(&buttons[SHOW_FUNCTIONS_BUTTON_INDEX], selectedButton == &buttons[SHOW_FUNCTIONS_BUTTON_INDEX]);
		Button_draw(&buttons[ANIMATE_BUTTON_INDEX], selectedButton == &buttons[ANIMATE_BUTTON_INDEX]);
		drawGlutString(5, 128, "These values can be modified by dragging up or down with your mouse.");
		drawGlutString(5, 114, "For greater precision, start the drag closer to the left side of the field.");
		drawGlutString(5, 101, "While dragging, type a number (0-9) to set the value of the field. You can also negate it with -.");
	}
	
	drawGlutString(viewport[2] / 2 - 204, viewport[3] - 13, "The center object represents the quaternion itself.");
	drawGlutString(viewport[2] / 2 - 256, viewport[3] - 26, "The objects on the left and right are rotated by the quaternion.");
	drawGlutString(viewport[2] / 2 - 300, viewport[3] - 39, "You can drag your mouse in the scene to apply a rotation to the quaternion.");
	
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glEnable(GL_LIGHTING);
	glEnable(GL_DEPTH_TEST);
	
	glutSwapBuffers();
	
	if (slerpStartTime != 0) {
		float alpha;
		
		alpha = (currentTime - slerpStartTime) / 500.0f;
		if (alpha >= 1.0f) {
			alpha = 1.0f;
			slerpStartTime = 0;
		}
		quaternion = Quaternion_slerp(slerpStartQuaternion, slerpEndQuaternion, alpha);
		QuaternionControls_setQuaternion(quaternion);
		
		glutPostRedisplay();
		
	} else if (animate) {
		glutPostRedisplay();
	}
}

void reshapeFunc(int width, int height) {
	glViewport(0, 0, width, height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(60, (float) width / (float) height, 1.5, 400.0);
	glMatrixMode(GL_MODELVIEW);
#ifndef __APPLE__
	displayFunc();
#endif
}

void mouseFunc(int button, int state, int x, int y) {
	if (button == GLUT_LEFT_BUTTON) {
		GLint viewport[4];
		
		glGetIntegerv(GL_VIEWPORT, viewport);
		y = viewport[3] - y;
		
		mouseIsDown = state == GLUT_DOWN;
		lastMouseX = x;
		lastMouseY = y;
		
		if (mouseIsDown) {
			if (showingFunctions) {
				if (FunctionControls_mouseDown(x, y)) {
					draggingFunctionField = true;
					
				} else if (Button_hitTest(&buttons[RESET_TO_IDENTITY_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[RESET_TO_IDENTITY_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[SHOW_FUNCTIONS_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[SHOW_FUNCTIONS_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[ANIMATE_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[ANIMATE_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[APPLY_NORMALIZE_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[APPLY_NORMALIZE_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[APPLY_INVERT_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[APPLY_INVERT_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[APPLY_MULTIPLY_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[APPLY_MULTIPLY_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[APPLY_SLERP_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[APPLY_SLERP_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[APPLY_ROTATE_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[APPLY_ROTATE_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[CURRENT_MULTIPLY_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[CURRENT_MULTIPLY_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[CURRENT_SLERP_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[CURRENT_SLERP_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[CURRENT_ROTATE_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[CURRENT_ROTATE_BUTTON_INDEX];
				}
			} else {
				if (QuaternionControls_mouseDown(x, y)) {
					draggingNumberField = true;
					
				} else if (Button_hitTest(&buttons[AUTO_NORMALIZE_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[AUTO_NORMALIZE_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[RESET_TO_IDENTITY_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[RESET_TO_IDENTITY_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[SHOW_FUNCTIONS_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[SHOW_FUNCTIONS_BUTTON_INDEX];
					
				} else if (Button_hitTest(&buttons[ANIMATE_BUTTON_INDEX], x, y)) {
					selectedButton = &buttons[ANIMATE_BUTTON_INDEX];
				}
			}
		} else {
			if (draggingNumberField) {
				QuaternionControls_mouseUp(x, y);
				draggingNumberField = false;
				
			} else if (draggingFunctionField) {
				FunctionControls_mouseUp(x, y);
				draggingFunctionField = false;
				
			} else if (selectedButton != NULL) {
				if (Button_hitTest(selectedButton, x, y)) {
					if (selectedButton == &buttons[AUTO_NORMALIZE_BUTTON_INDEX]) {
						autoNormalize = !autoNormalize;
						buttons[AUTO_NORMALIZE_BUTTON_INDEX].label = (autoNormalize ? "Auto-normalize ON" : "Auto-normalize OFF");
						if (autoNormalize) {
							Quaternion_normalize(&quaternion);
						}
						QuaternionControls_setQuaternion(quaternion);
						
					} else if (selectedButton == &buttons[RESET_TO_IDENTITY_BUTTON_INDEX]) {
						quaternion = Quaternion_identity();
						QuaternionControls_setQuaternion(quaternion);
						
					} else if (selectedButton == &buttons[SHOW_FUNCTIONS_BUTTON_INDEX]) {
						showingFunctions = !showingFunctions;
						buttons[SHOW_FUNCTIONS_BUTTON_INDEX].label = (showingFunctions ? "Numbers ->" : "Functions ->");
						
					} else if (selectedButton == &buttons[ANIMATE_BUTTON_INDEX]) {
						animate = !animate;
						buttons[ANIMATE_BUTTON_INDEX].label = (animate ? "Animate ON" : "Animate OFF");
						objectQuaternion = quaternion;
						animateStartTime = glutGet(GLUT_ELAPSED_TIME);
						
					} else if (selectedButton == &buttons[APPLY_NORMALIZE_BUTTON_INDEX]) {
						Quaternion_normalize(&quaternion);
						QuaternionControls_setQuaternion(quaternion);
						
					} else if (selectedButton == &buttons[APPLY_INVERT_BUTTON_INDEX]) {
						Quaternion_invert(&quaternion);
						QuaternionControls_setQuaternion(quaternion);
						
					} else if (selectedButton == &buttons[APPLY_MULTIPLY_BUTTON_INDEX]) {
						slerpStartQuaternion = quaternion;
						slerpEndQuaternion = FunctionControls_multiply(quaternion);
						slerpStartTime = glutGet(GLUT_ELAPSED_TIME);
						
					} else if (selectedButton == &buttons[APPLY_SLERP_BUTTON_INDEX]) {
						slerpStartQuaternion = quaternion;
						slerpEndQuaternion = FunctionControls_slerp(quaternion);
						slerpStartTime = glutGet(GLUT_ELAPSED_TIME);
						
					} else if (selectedButton == &buttons[APPLY_ROTATE_BUTTON_INDEX]) {
						slerpStartQuaternion = quaternion;
						slerpEndQuaternion = FunctionControls_rotate(quaternion);
						slerpStartTime = glutGet(GLUT_ELAPSED_TIME);
						
					} else if (selectedButton == &buttons[CURRENT_MULTIPLY_BUTTON_INDEX]) {
						FunctionControls_setMultiplyQuaternion(quaternion);
						
					} else if (selectedButton == &buttons[CURRENT_SLERP_BUTTON_INDEX]) {
						FunctionControls_setSLERPQuaternion(quaternion);
						
					} else if (selectedButton == &buttons[CURRENT_ROTATE_BUTTON_INDEX]) {
						Vector axis;
						float angle;
						
						Quaternion_toAxisAngle(quaternion, &axis, &angle);
						FunctionControls_setRotateAxisAngle(axis, angle);
					}
				}
				selectedButton = NULL;
			}
		}
		glutPostRedisplay();
	}
}

#define ROTATION_RADIANS_PER_PIXEL (M_PI / 400.0)

void motionFunc(int x, int y) {
	if (mouseIsDown) {
		GLint viewport[4];
		int deltaX, deltaY;
		
		glGetIntegerv(GL_VIEWPORT, viewport);
		y = viewport[3] - y;
		
		deltaX = x - lastMouseX;
		deltaY = y - lastMouseY;
		lastMouseX = x;
		lastMouseY = y;
		
		if (draggingNumberField) {
			QuaternionControls_mouseDragged(deltaX, deltaY);
			
		} else if (draggingFunctionField) {
			FunctionControls_mouseDragged(deltaX, deltaY);
			
		} else if (selectedButton == NULL) {
			Quaternion inverseQuaternion;
			Vector up = {0.0f, 1.0f, 0.0f}, right = {-1.0f, 0.0f, 0.0f};
			
			inverseQuaternion = Quaternion_inverted(quaternion);
			up = Quaternion_multiplyVector(inverseQuaternion, up);
			Quaternion_rotate(&quaternion, up, deltaX * ROTATION_RADIANS_PER_PIXEL);
			
			inverseQuaternion = Quaternion_inverted(quaternion);
			right = Quaternion_multiplyVector(inverseQuaternion, right);
			Quaternion_rotate(&quaternion, right, deltaY * ROTATION_RADIANS_PER_PIXEL);
			
			QuaternionControls_setQuaternion(quaternion);
			glutPostRedisplay();
		}
	}
}

void keyboardFunc(unsigned char key, int x, int y) {
	if (key == 'r') {
		quaternion = Quaternion_identity();
		QuaternionControls_setQuaternion(quaternion);
		glutPostRedisplay();
	} else if (key == ' ') {
		if (draggingNumberField) {
			QuaternionControls_setEditFieldValue(0.0f);
		} else if (draggingFunctionField) {
			FunctionControls_setEditFieldValue(0.0f);
		}
	} else if (key >= '0' && key <= '9') {
		if (draggingNumberField) {
			QuaternionControls_setEditFieldValue(key - '0');
		} else if (draggingFunctionField) {
			FunctionControls_setEditFieldValue(key - '0');
		}
	} else if (key == '-') {
		if (draggingNumberField) {
			QuaternionControls_setEditFieldValue(-QuaternionControls_getEditFieldValue());
		} else if (draggingFunctionField) {
			FunctionControls_setEditFieldValue(-FunctionControls_getEditFieldValue());
		}
	}
}

int main(int argc, char ** argv) {
	glutInit(&argc, argv);
	
	glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);
	glutInitWindowPosition(2, 28);
	glutInitWindowSize(1000, 600);
	glutCreateWindow("Quaternion playground");
	
	glutReshapeFunc(reshapeFunc);
	glutDisplayFunc(displayFunc);
	
	glutMouseFunc(mouseFunc);
	glutMotionFunc(motionFunc);
	glutKeyboardFunc(keyboardFunc);
	
	initGL();
	initPlayground();
	
	glutMainLoop();
	
	return EXIT_SUCCESS;
}
